import XCTest
import Foundation

/// Centralized test configuration for the LifeClock Watch App test suite
/// Provides common test utilities, mock data, and configuration settings
class TestConfiguration {
    
    // MARK: - Test Coverage Requirements
    static let minimumCodeCoverage: Double = 0.80 // 80% minimum
    
    // MARK: - Mock Data Configuration
    struct MockHealthData {
        static let sampleBirthDate = Calendar.current.date(from: DateComponents(year: 1985, month: 6, day: 15))!
        static let sampleHeight: Double = 175.0 // cm
        static let sampleWeight: Double = 70.0 // kg
        static let sampleHeartRate: Double = 72.0 // bpm
        static let sampleSteps: Double = 8500.0 // daily steps
        static let sampleSleepHours: Double = 7.5 // hours per night
    }
    
    struct MockSurveyData {
        static let sampleSmokingStatus = false
        static let sampleExerciseFrequency = 4 // times per week
        static let sampleStressLevel = 3 // scale 1-5
        static let sampleDietQuality = 4 // scale 1-5
    }
    
    // MARK: - Test Timeouts
    static let standardTimeout: TimeInterval = 5.0
    static let networkTimeout: TimeInterval = 10.0
    static let performanceTimeout: TimeInterval = 1.0
    
    // MARK: - Performance Benchmarks
    struct PerformanceBenchmarks {
        static let actuarialCalculationTime: TimeInterval = 0.1 // 100ms max
        static let databaseQueryTime: TimeInterval = 0.05 // 50ms max
        static let uiRenderTime: TimeInterval = 0.016 // 60fps target
        static let memoryUsageLimit: Int = 50_000_000 // 50MB max
    }
    
    // MARK: - Test Database Configuration
    static func createTestDatabase() -> URL {
        let tempDirectory = FileManager.default.temporaryDirectory
        let testDBURL = tempDirectory.appendingPathComponent("test_lifeclock.sqlite")
        
        // Clean up existing test database
        try? FileManager.default.removeItem(at: testDBURL)
        
        return testDBURL
    }
    
    // MARK: - Common Test Utilities
    static func generateRandomHealthProfile() -> (age: Int, bmi: Double, riskFactors: [String]) {
        let age = Int.random(in: 18...90)
        let height = Double.random(in: 150...200) // cm
        let weight = Double.random(in: 50...120) // kg
        let bmi = weight / pow(height / 100, 2)
        
        var riskFactors: [String] = []
        if bmi > 30 { riskFactors.append("obesity") }
        if age > 65 { riskFactors.append("advanced_age") }
        if Bool.random() { riskFactors.append("smoking") }
        
        return (age, bmi, riskFactors)
    }
    
    static func createMockPredictionResult() -> (lifeExpectancy: Double, confidenceInterval: (lower: Double, upper: Double)) {
        let baseLife = Double.random(in: 65...95)
        let variance = Double.random(in: 5...15)
        return (baseLife, (baseLife - variance, baseLife + variance))
    }
    
    // MARK: - Assertion Helpers
    static func assertWithinTolerance(_ actual: Double, _ expected: Double, tolerance: Double = 0.01, file: StaticString = #file, line: UInt = #line) {
        XCTAssertEqual(actual, expected, accuracy: tolerance, file: file, line: line)
    }
    
    static func assertPerformance<T>(_ block: () throws -> T, maxExecutionTime: TimeInterval, file: StaticString = #file, line: UInt = #line) rethrows -> T {
        let startTime = CFAbsoluteTimeGetCurrent()
        let result = try block()
        let executionTime = CFAbsoluteTimeGetCurrent() - startTime
        
        XCTAssertLessThan(executionTime, maxExecutionTime, "Execution took \(executionTime)s, expected < \(maxExecutionTime)s", file: file, line: line)
        return result
    }
    
    // MARK: - Test Data Cleanup
    static func cleanupTestData() {
        // Clean up temporary files
        let tempDirectory = FileManager.default.temporaryDirectory
        let testFiles = ["test_lifeclock.sqlite", "test_predictions.json", "test_health_data.plist"]
        
        for filename in testFiles {
            let fileURL = tempDirectory.appendingPathComponent(filename)
            try? FileManager.default.removeItem(at: fileURL)
        }
    }
    
    // MARK: - Mock Network Responses
    struct MockNetworkData {
        static let healthKitAuthResponse = """
        {
            "authorized": true,
            "permissions": ["height", "weight", "heart_rate", "steps", "sleep"]
        }
        """
        
        static let predictionResponse = """
        {
            "life_expectancy": 82.5,
            "confidence_lower": 78.2,
            "confidence_upper": 86.8,
            "risk_factors": ["moderate_bmi", "good_exercise"],
            "calculation_date": "2025-09-16T10:00:00Z"
        }
        """
    }
    
    // MARK: - Accessibility Testing
    static func validateAccessibility(for view: Any) {
        // Helper function to validate basic accessibility requirements
        // This would be expanded based on actual SwiftUI/UIKit implementation
        print("Accessibility validation for: \(type(of: view))")
    }
    
    // MARK: - Test Environment Setup
    static func setupTestEnvironment() {
        // Configure test-specific settings
        UserDefaults.standard.set(true, forKey: "isTestEnvironment")
        UserDefaults.standard.set(false, forKey: "hasCompletedOnboarding")
        UserDefaults.standard.set(false, forKey: "hasAcceptedDisclaimer")
    }
    
    static func teardownTestEnvironment() {
        // Clean up test environment
        let testKeys = ["isTestEnvironment", "hasCompletedOnboarding", "hasAcceptedDisclaimer"]
        for key in testKeys {
            UserDefaults.standard.removeObject(forKey: key)
        }
        cleanupTestData()
    }
}

// MARK: - Test Extensions
extension XCTestCase {
    
    func measurePerformance<T>(description: String = "", block: () throws -> T) rethrows -> T {
        return try TestConfiguration.assertPerformance(block, maxExecutionTime: TestConfiguration.PerformanceBenchmarks.actuarialCalculationTime)
    }
    
    func setupTest() {
        TestConfiguration.setupTestEnvironment()
    }
    
    func teardownTest() {
        TestConfiguration.teardownTestEnvironment()
    }
}